/**
 * A class for complete labelled binary trees
 *
 * @version $Id: BinaryTree.java,v 1.1.2.1 2003/10/30 00:03:08 nthiery Exp $
 * @author  Nicolas M. Thiry <nthiery@users.sourceforge.net>
**/

import java.util.LinkedList;
import java.util.ListIterator;

public class BinaryTree {

    Object label;
    BinaryTree leftChild;
    BinaryTree rightChild;

    // Invariant:
    // leftChild and rightChild are either both null or both binary trees

    /**
     * Constructor for a tree with a single node
     **/
    public BinaryTree(Object l) {
	label = l;
	leftChild=null;
	rightChild=null;
    }

    /**
     * Constructor for a tree with two subtrees
     **/
    public BinaryTree(Object l, BinaryTree left, BinaryTree right) {
	label = l;
	leftChild = left;
	rightChild = right;
    }

    /**
     * Returns the number of nodes of this tree
     *
     * @return the number of nodes of this tree
     **/
    public int size () {
	if (leftChild == null) {
	    return 1;
	} else {
	    return 1 + leftChild.size() + rightChild.size();
	}
    }

    /*
      Note: the implementations of the three methods
      toArrayInfix/Prefix/Postfix are almost exactly the same. To
      avoid duplicated code, we implement a single generic method that
      takes a parameter to specify the desired ordering.
     */
    
    /**
     * Returns the labels contained in this tree, in infix order
     *
     * @return the labels contained in this tree, in infix order
     **/
    public Object[] toArrayInfix() {
	return toArray("infix");
    }

    /**
     * Returns the labels contained in this tree, in prefix order
     *
     * @return the labels contained in this tree, in prefix order
     **/
    public Object[] toArrayPrefix() {
	return toArray("prefix");
    }

    /**
     * Returns the labels contained in this tree, in postfix order
     *
     * @return the labels contained in this tree, in postfix order
     **/
    public Object[] toArrayPostfix() {
	return toArray("postfix");
    }

    /**
     * Returns the labels contained in this tree, in the order
     * specified by the argument s.
     *
     * @param s  "prefix", "infix", or "postfix"
     * @return the labels contained in this tree, in infix order
     **/

    /*
      Algorithm: we create once for all an array `result` for all the
      labels, and pass it by reference to a recursive method that
      traverse the tree and fills progressively result.
     */
    public Object[] toArray(String s) {
	Object[] result = new Object[this.size()];
	toArrayRec(s, result, 0);
	return result;
    }

    /**
     * Writes the labels contained in this tree in the array result
     * (which is passed by reference), starting at position j, and
     * using the order specified by s
     *
     * @param result	an array
     * @param j         a position in result; there should be at least
     *                  this.size() free positions in result starting
     *                  from position j
     * @param s		"prefix", "infix", or "postfix"
     * @return		the next available position in result after the operation
     **/
    private int toArrayRec(String s, Object[] result, int j) {
	if (leftChild == null) {
	    // Leave
	    result[j] = label;
	    j++;
	    return j;
	}
	
	// Internal node:
	if (s.equals("prefix")) {
	    result[j] = label;
	    j++;
	}
	j = leftChild.toArrayRec(s, result, j);
	if (s.equals("infix")) {
	    result[j] = label;
	    j++;
	}
	j = rightChild.toArrayRec(s, result, j);
	if (s.equals("postfix")) {
	    result[j] = label;
	    j++;
	}
	return(j);
    }

    /**
     * Returns the labels contained in this tree, in breath first order
     *
     * @return the labels contained in this tree, in breath first order
     **/
    public Object[] toArrayBreathFirst() {
	// Proceed level by level
	// level contains all the nodes (actually binary trees)
	// at the current level
	LinkedList level = new LinkedList();
	level.add(this);
	Object[] result = new Object[size()];
	int j=0; // Invariant: the positions 0..j-i of result have been filled
	while(level.size()!=0) {
	    // At the end of the loop, newLevel contains all the
	    // nodes at the next level
	    LinkedList newLevel = new LinkedList();

	    // Iterate through the nodes in level.
	    // we could use a simple loop here, but it's more
	    // efficient to use a linked list and an iterator
	    for (ListIterator it=level.listIterator(0); it.hasNext();j++) {
		BinaryTree t = (BinaryTree)it.next();
		result[j] = t.label;
		if (t.leftChild!=null) {
		    newLevel.add(t.leftChild);
		    newLevel.add(t.rightChild);
		}
		level=newLevel;
	    }
	}
	return result;
    }

}
